/******************************************************************************
 *
 * Copyright (c) 2004 PalmSource, Inc. All rights reserved.
 *
 * File: DevNub.c
 *
 * Release: Palm OS Developer Suite 5 SDK (68K) 4.0
 *
 * Description:
 *
 * This is source for a client-side module that assists in the develoment of
 * Palm OS applications.  After it is installed on the client (or "debug
 * target"), it provides the following:
 *
 * 	*	DotDotTwo functionality.  DotDotTwo is a utility application provided
 * 		by PalmSource to manage the launching of the "debug console" built
 * 		into Palm devices.  The debug console allows external facilities such
 * 		as development environments to connect to the debug target and
 * 		manipulate it.
 *
 * 	*	Sub-launching functionality.  Some applications are developed to be
 * 		"sub-launched", meaning that they are called as subroutines of some
 * 		other process and not really launched as a proper process themselves.
 * 		DevNub serves as the controlling process when the user opts to debug
 * 		their application in such a context.
 *
 *	*	"Booting is done" notification.  Debugging an application often
 * 		results in rebooting the target device when the debugging session is
 * 		over.  While rebooting, the user cannot start a new debug session.
 * 		DevNub provides notification allowing the development environment to
 * 		determine if the booting process is completed and a new debug session
 * 		can be started.
 *
 *	*	Pen calibration.  Palm OS Simulator asks the user to calibrate the
 * 		"touchscreen" when it performs a cold boot (as it does when starting
 * 		up after being launched).  This calibration is not needed, and can be
 * 		set to some identity matrix.  DevNub provides that setup.
 *
 *****************************************************************************/

#include <PalmOS.h>

#define UNUSED  __attribute__((__unused__))

#define NOTIFY_DEBUG_CONSOLE 0


/******************************************************************************
 *
 *	Internal Structures
 *
 *****************************************************************************/

typedef struct DevNubPreferenceType
{
	Boolean	enterDebugConsoleOnReset;
	Boolean enteredDebugConsole;
	Boolean serialInUse;
} DevNubPreferenceType;


/******************************************************************************
 *
 *	Global variables
 *
 *****************************************************************************/

DevNubPreferenceType	gPrefs;


/******************************************************************************
 *
 *	Internal Constants
 *
 *****************************************************************************/

#define appFileCreator			'devn'
#define appVersionNum			1
#define appPrefID				0
#define appPrefVersionNum		1

#if NOTIFY_DEBUG_CONSOLE
#define onResetAlarmRef			1
#define onResetAlarmDelay		3	// 3 seconds is longer than the VFS manager's own alarm
#define cardNoFeature			1
#define dbIdFeature				2
#define resetFinishedPriority	92	// 92 is a very large priority, so this app will be notifed very late.
#define displayWarningLaunchCmd	sysAppLaunchCmdCustomBase
#endif

#define sysAppLaunch			((UInt16) (sysAppLaunchCmdCustomBase + 1))
#define sysUIAppSwitch			((UInt16) (sysAppLaunchCmdCustomBase + 2))


/******************************************************************************
 *
 *	Resource Constants
 *
 *****************************************************************************/

//	Resource: Talt 1000

#define ResetOptionAlert		1000
#define PortInUseAlert			1100
#define OnResetAlert			1200
#define OnPoserAlert			1300

//	Resource: tFRM 1000

#define MainForm				1000
#define MainEnterNow			1002
#define MainEnterOnReset		1003
#define MainCurrentStatusLabel	1005

//	Resource: tSTL 1000

#define StatusStringList		1000


/******************************************************************************
 *
 * FUNCTION:	Read16
 *
 * DESCRIPTION:	Read a 16-bit value in Big Endian format from the given offset
 *				from the given pointer.
 *
 * PARAMETERS:	p - pointer to the buffer containing the data
 *				offset - offset to the value to extract
 *
 * RETURNED:	The 16-bit value.
 *
 *****************************************************************************/

static UInt16 Read16(void const* p, int offset)
{
	UInt8 const* p2 = (UInt8 const*) p;

	return
		(((UInt16) p2[offset + 0]) << 8) |
		(((UInt16) p2[offset + 1]) << 0);
}


/******************************************************************************
 *
 * FUNCTION:	Read32
 *
 * DESCRIPTION:	Read a 32-bit value in Big Endian format from the given offset
 *				from the given pointer.
 *
 * PARAMETERS:	p - pointer to the buffer containing the data
 *				offset - offset to the value to extract
 *
 * RETURNED:	The 32-bit value.
 *
 *****************************************************************************/

static UInt32 Read32(void const* p, int offset)
{
	UInt8 const* p2 = (UInt8 const*) p;

	return
		(((UInt32) p2[offset + 0]) << 24) |
		(((UInt32) p2[offset + 1]) << 16) |
		(((UInt32) p2[offset + 2]) << 8) |
		(((UInt32) p2[offset + 3]) << 0);
}


/******************************************************************************
 *
 * FUNCTION:	ReallocateCmdPBP
 *
 * DESCRIPTION:	Form a new command parameter block from the existing one,
 *				using the data starting at the given offset into the existing
 *				one and extending to the end of the existing one.
 *
 * PARAMETERS:	cmdPBP - pointer the existing parameter block
 *				offset - offset into the existing block where to find the data
 *					for the new parameter block
 *
 * RETURNED:	The new parameter block.  Will be -1 if a new block of the
 *				required size cannot be allocated, or NULL if the offset is
 *				past the end of the existing block.
 *
 *****************************************************************************/

static MemPtr ReallocateCmdPBP(MemPtr cmdPBP, UInt32 offset)
{
	UInt32	cmdPBPSize	= MemPtrSize(cmdPBP);
	MemPtr	subCmdPBP	= NULL;

	if (cmdPBPSize > offset)
	{
		subCmdPBP = MemPtrNew(cmdPBPSize - offset);
		if (subCmdPBP == NULL)
		{
			return (MemPtr) -1;
		}

		MemPtrSetOwner(subCmdPBP, 0);
		MemMove(subCmdPBP, ((char*) cmdPBP) + offset, cmdPBPSize - offset);
	}

	return subCmdPBP;
}


/******************************************************************************
 *
 * FUNCTION:	PalmOSVersion
 *
 * DESCRIPTION:	Return the Palm OS version number.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	The OS version number, in sysMakeROMVersion format.  Returns
 *				zero if for some reason the OS version can't be read.
 *
 *****************************************************************************/

static UInt32 PalmOSVersion(void)
{
	UInt32	palmOSVersion;
	if (FtrGet(sysFtrCreator, sysFtrNumROMVersion, &palmOSVersion) != errNone)
	{
		palmOSVersion = 0;  // Set to very low version if FtrGet failed.
	}

	return palmOSVersion;
}


/******************************************************************************
 *
 * FUNCTION:	OnPoser
 *
 * DESCRIPTION:	Return whether or not we're running under the Palm OS Emulator.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	True if we're running under the Palm OS Emulator
 *
 *****************************************************************************/

static Boolean OnPoser(void)
{
	UInt32	value;

	// On Palm OS 3.0 or later, HostGetHostID is guaranteed to be implemented.

	if (sysGetROMVerMajor(PalmOSVersion()) >= 0x03)
	{
		return HostGetHostID() == hostIDPalmOSEmulator;
	}

	// On earlier systems, check for a System Feature that Poser automatically
	// installs during bootup.

	return FtrGet(kPalmOSEmulatorFeatureCreator,
		kPalmOSEmulatorFeatureNumber, &value) == errNone;
}


/******************************************************************************
 *
 * FUNCTION:	OnPalmSim
 *
 * DESCRIPTION:	Return whether or not we're running under the Palm OS Simulator.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	True if we're running under the Palm OS Simulator
 *
 *****************************************************************************/

static Boolean OnPalmSim(void)
{
	// On Palm OS 3.0 or later, HostGetHostID is guaranteed to be implemented.

	if (sysGetROMVerMajor(PalmOSVersion()) >= 0x03)
	{
		return HostGetHostID() == hostIDPalmOSSimulator;
	}

	// On Palm OS 2.x, we can call SysTicksPerSecond and return true if it's
	// 100 (a Mac Simulator-specific value).

	if (sysGetROMVerMajor(PalmOSVersion()) >= 0x02)
	{
		return SysTicksPerSecond() == 100;
	}

	// No good test for Palm OS 1.0?

	return false;
}


/******************************************************************************
 *
 * FUNCTION:	OnPalmOSCobalt
 *
 * DESCRIPTION:	Return whether or not we're running under Palm OS Cobalt.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	True if we're running under Palm OS Cobalt
 *
 *****************************************************************************/

static Boolean OnPalmOSCobalt(void)
{
	return (sysGetROMVerMajor(PalmOSVersion()) >= 0x06);
}


/******************************************************************************
 *
 * FUNCTION:	GetObjectPtr
 *
 * DESCRIPTION:	This routine returns a pointer to an object in the current
 *				form.
 *
 * PARAMETERS:	objectID - id of the object to find
 *
 * RETURNED:	Pointer to the found object
 *
 *****************************************************************************/

static void* GetObjectPtr(FormPtr frmP, UInt16 objectID)
{
	UInt16 index = FrmGetObjectIndex(frmP, objectID);
	ErrFatalDisplayIf(index == frmInvalidObjectId, "Invalid form object ID");
	return FrmGetObjectPtr(frmP, index);
}


/******************************************************************************
 *
 * FUNCTION:	AllowProscribedCall
 *
 * DESCRIPTION:	Configure our environment to allow calls to system functions
 *				that would otherwise be reported as proscribed.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	The previous setting of the option
 *
 *****************************************************************************/

static int AllowProscribedCall(void)
{
	if ((OnPoser() || OnPalmSim()) &&
		HostIsSelectorImplemented(hostSelectorGetPreference))
	{
		char			result[20];
		HostBoolType	exists;

		exists = HostGetPreference("ReportProscribedFunction", result);
		HostSetPreference("ReportProscribedFunction", "0");
		return exists && (strcmp(result, "0") != 0);
	}

	return 0;
}


/******************************************************************************
 *
 * FUNCTION:	RestoreProscribedCall
 *
 * DESCRIPTION:	Set the option controlling the reporting of proscribed
 *				function calls to the given value (typically the previous
 *				value as returned by AllowProscribedCall).
 *
 * PARAMETERS:	The value to which to set the option
 *
 * RETURNED:	Nothing
 *
 *****************************************************************************/

static void RestoreProscribedCall(int previous)
{
	if ((OnPoser() || OnPalmSim()) &&
		HostIsSelectorImplemented(hostSelectorGetPreference))
	{
		HostSetPreference("ReportProscribedFunction", previous ? "1" : "0");
	}
}


/******************************************************************************
 *
 * FUNCTION:	GetPrefs
 *
 * DESCRIPTION:	Fetches the current application's preferences.
 *
 * PARAMETERS:	Pointer to a DevNubPreferenceType structure.  This structure
 *				receives the preference values.
 *
 * RETURNED:	Nothing
 *
 *****************************************************************************/

static void GetPrefs(DevNubPreferenceType* pPrefs)
{
	// Read the saved preferences / saved-state information.

	UInt16 prefsSize = sizeof(*pPrefs);

	ErrFatalDisplayIf(OnPoser(), "Should not get here on Poser");

	if (PrefGetAppPreferences(
			appFileCreator, appPrefID, pPrefs, &prefsSize, true)
		== noPreferenceFound)
	{
		// Set default preferences.

		pPrefs->enterDebugConsoleOnReset = true;
		pPrefs->enteredDebugConsole = false;
		pPrefs->serialInUse = false;
	}
}


/******************************************************************************
 *
 * FUNCTION:	SavePrefs
 *
 * DESCRIPTION:	Saves the current application's preferences.
 *
 * PARAMETERS:	Pointer to a DevNubPreferenceType structure.  This structure
 *				provides the values to be saved.
 *
 * RETURNED:	Nothing
 *
 *****************************************************************************/

static void SavePrefs(DevNubPreferenceType* pPrefs)
{
	PrefSetAppPreferences(appFileCreator, appPrefID, appPrefVersionNum,
		pPrefs, sizeof(*pPrefs), true);
}


/******************************************************************************
 *
 * FUNCTION:	EnterConsoleMode
 *
 * DESCRIPTION:	Enters console mode and records the results in the
 *				preferences.
 *
 * PARAMETERS:	Pointer to a DevNubPreferenceType structure.  This structure
 *				contains the preference regarding whether or not the user
 *				wants to start the debug console, and receives the results
 *				of the attempt to start it.
 *
 * RETURNED:	Nothing
 *
 *****************************************************************************/

static void EnterConsoleMode(DevNubPreferenceType* pPrefs)
{
	ErrFatalDisplayIf(OnPoser(), "Should not get here on Poser");

	// If the device is already in console mode, do nothing.

	if (!pPrefs->enteredDebugConsole)
	{
		Err	err;
		int	previous;

		previous = AllowProscribedCall();
		err = SysLaunchConsole();
		RestoreProscribedCall(previous);

		switch (err)
		{
			case errNone:
				pPrefs->enteredDebugConsole = true;
				pPrefs->serialInUse = false;
				break;

			case serErrAlreadyOpen:
				pPrefs->enteredDebugConsole = false;
				pPrefs->serialInUse = true;
				break;

			default:
				pPrefs->enteredDebugConsole = false;
				pPrefs->serialInUse = false;
				break;
		}

		SavePrefs(pPrefs);
	}
}


/******************************************************************************
 *
 * FUNCTION:	ResetCalibrationInfo
 *
 * DESCRIPTION: Sets the pen calibration info to be "perfect": no
 *				translation or scaling.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 *****************************************************************************/

void ResetCalibrationInfo(void)
{
	#define target0X	10				// top left
	#define target0Y	10
	#define target1X	(160-10)		// bottom right
	#define target1Y	(160-10)

	Err			err;
	PointType	scrPoints[2];

	ErrFatalDisplayIf(OnPoser(), "Should not get here on Poser");

	scrPoints[0].x = target0X;
	scrPoints[0].y = target0Y;
	scrPoints[1].x = target1X;
	scrPoints[1].y = target1Y;

	// Reset the calibration info in the Pen Mgr's globals to an
	// identity state.

	err = PenResetCalibration();

	// In order to make those settings "stick", we need to call
	// PenCalibrate.  This will write the settings to the Pen
	// Mgr's preferences in the preferences database.  Ensuring
	// that these preferences exist will also prevent the digitizer
	// panel from being run at boot up on older OSes (current OSes
	// check for Poser and inhibit it directly).
	//
	// By passing in the same points for the "screen" and "digitizer"
	// values, we maintain the identity state.

	err = PenCalibrate(
		&scrPoints[0], &scrPoints[1],
		&scrPoints[0], &scrPoints[1]);
}


/******************************************************************************
 *
 * FUNCTION:	ResetFinishedCallback
 *
 * DESCRIPTION:	Receives the notification that the device has finished
 *				resetting
 *
 * PARAMETERS:	see the documentation for SysNotifyRegister
 *
 * RETURNED:	Error code
 *
 *****************************************************************************/

#if NOTIFY_DEBUG_CONSOLE
static Err ResetFinishedCallback(SysNotifyParamType* notifyParamsP)
{
	switch (notifyParamsP->notifyType)
	{
		case sysNotifyResetFinishedEvent:
		{
			UInt32 cardNo;
			UInt32 dbID;

			// A reset has finished. It'd be nice to notify the user at this
			// point, but there's very little stack space available during
			// this notification. Instead, set an alarm that will be
			// triggered in just a few seconds.

			// We need the cardNo and dbID of our app to set the alarm, but
			// SysCurAppDatabase() will give us incorrect information because
			// this is just a callback notification. Instead, use the
			// information we stored earlier in features.

			FtrGet(appFileCreator, cardNoFeature, &cardNo);
			FtrUnregister(appFileCreator, cardNoFeature);
			FtrGet(appFileCreator, dbIdFeature, &dbID);
			FtrUnregister(appFileCreator, dbIdFeature);

			AlmSetAlarm(cardNo, dbID, onResetAlarmRef,
				TimGetSeconds() + onResetAlarmDelay, false);

			return errNone;
		}
  }

	return errNone;
}
#endif


/******************************************************************************
 *
 * FUNCTION:	AppStart
 *
 * DESCRIPTION:	Get the current application's preferences and scope out the
 *				device we're running on.  Show an alert if we tried to start
 *				the debug console and failed.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Err value 0 if nothing went wrong
 *
 *****************************************************************************/

static Err AppStart(void)
{
	// Read the saved preferences / saved-state information.

	GetPrefs(&gPrefs);

	// On Palm OS Cobalt, the console does not stay open after this application
	// opens it during reset and then exits.  Therefore, we have to open it
	// again here when the application is launched normally by hand.

	if (OnPalmOSCobalt())
	{
		EnterConsoleMode(&gPrefs);
		if (gPrefs.serialInUse)
		{
			FrmAlert(PortInUseAlert);
		}
	}

	return errNone;
}


/******************************************************************************
 *
 * FUNCTION:	AppStop
 *
 * DESCRIPTION:	Save the current state of the application.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 *****************************************************************************/

static void AppStop(void)
{
	// Write the saved preferences / saved-state information.  This data
	// will saved during a HotSync backup.

	SavePrefs(&gPrefs);
	    
	// Close all the open forms.  We shouldn't have any, but this is a good
	// practice anyway.

	FrmCloseAllForms();
}


/******************************************************************************
 *
 * FUNCTION:	MainFormHandleEvent
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	.
 *
 * RETURNED:	.
 *
 *****************************************************************************/

static Boolean MainFormHandleEvent(EventType* eventP)
{
	switch (eventP->eType)
	{
		case ctlSelectEvent:
		{
			UInt16	id = eventP->data.ctlSelect.controlID;
			
			if (id == MainEnterNow)
			{
				// TODO Warn that you can't turn this off.
				ControlType* checkbox = (ControlType*) eventP->data.ctlSelect.pControl;
				CtlSetValue(checkbox, true);

				EnterConsoleMode(&gPrefs);
				if (gPrefs.serialInUse)
				{
					FrmAlert(PortInUseAlert);
				}

				return true;
			}
			else if (id == MainEnterOnReset)
			{
				gPrefs.enterDebugConsoleOnReset = eventP->data.ctlSelect.on;
				return true;
			}
		}
		
		default:
			break;
	}

	return false;
}


/******************************************************************************
 *
 * FUNCTION:	HandleNormalLaunch
 *
 * DESCRIPTION:	Handle the launch code telling us that we're being launched
 *				as a normal application.
 *
 * PARAMETERS:	cmd - word value specifying the launch code. 
 *				cmdPB - pointer to a structure that is associated with the
 *					launch code. 
 *				launchFlags - word value providing extra information about the
 *					launch.
 *
 * RETURNED:	Result of launch
 *
 *****************************************************************************/

static UInt32 HandleNormalLaunch(
	UInt16 cmd UNUSED, MemPtr cmdPBP UNUSED, UInt16 launchFlags UNUSED)
{
	FormType*		myForm;
	ControlType*	checkbox;
	UInt8			stringIndex;
	Char			status[25];
	Err				error;

	if (OnPoser())
	{
		FrmAlert(OnPoserAlert);
		return errNone;
	}

	error = AppStart();
	if (error)
		return error;

	// Initialize the form.

	myForm = FrmInitForm(MainForm);
	ErrFatalDisplayIf(myForm == NULL, "Unable to create form");

	FrmSetEventHandler(myForm, MainFormHandleEvent);

	// Set the checkboxes.

	checkbox = GetObjectPtr(myForm, MainEnterNow);
	CtlSetValue(checkbox, gPrefs.enteredDebugConsole);

	checkbox = GetObjectPtr(myForm, MainEnterOnReset);
	CtlSetValue(checkbox, gPrefs.enterDebugConsoleOnReset);

	// Set the status label.

	if (gPrefs.enteredDebugConsole) {
		stringIndex = 0; // "in console mode"
	} else if (gPrefs.serialInUse) {
		stringIndex = 2; // "error: serial port busy"
	} else {
		stringIndex = 1; // "error"
	}

	SysStringByIndex(StatusStringList, stringIndex, status, sizeof(status));
	FrmCopyLabel(myForm, MainCurrentStatusLabel, status);   

	// Run the dialog.

	FrmDoDialog(myForm);
	FrmDeleteForm(myForm);

	AppStop();

	return errNone;
}


/******************************************************************************
 *
 * FUNCTION:	HandleSystemReset
 *
 * DESCRIPTION:	Handle the launch code telling us that the system is being
 *				reset.  Here, we launch the debug console if that is what the
 *				user has specified.
 *
 * PARAMETERS:	cmd - word value specifying the launch code. 
 *				cmdPB - pointer to a structure that is associated with the
 *					launch code. 
 *				launchFlags - word value providing extra information about the
 *					launch.
 *
 * RETURNED:	Result of launch
 *
 *****************************************************************************/

static UInt32 HandleSystemReset(
	UInt16 cmd UNUSED, MemPtr cmdPBP UNUSED, UInt16 launchFlags UNUSED)
{
	if (!OnPoser())
	{
		DevNubPreferenceType prefs;

#if NOTIFY_DEBUG_CONSOLE
		UInt16 cardNo;
		LocalID dbID;
#endif

		GetPrefs(&prefs);

		// Reset the prefs structure.

		prefs.enteredDebugConsole = false;
		prefs.serialInUse = false;

		// Start up debug console if requested.  Note that we don't do this
		// on Palm OS Cobalt, as it has no effect: the console will be shut
		// down when we exit this application.

		if (!OnPalmOSCobalt() && prefs.enterDebugConsoleOnReset)
		{
			EnterConsoleMode(&prefs);

#if NOTIFY_DEBUG_CONSOLE
			// We need to warn the developer somehow about the consequences of
			// always being in console mode, but UI is not allowed during this
			// notification. A bunch of other applications are going to receive the
			// notification, so we'll register for the sysNotifyResetFinishedEvent
			// notification.

			SysCurAppDatabase(&cardNo, &dbID);

			// Save the information needed during the callback notification.

			FtrSet(appFileCreator, cardNoFeature, cardNo);
			FtrSet(appFileCreator, dbIdFeature, dbID);

			// Lock down the code resource so it is valid for a callback style
			// notification.

			MemHandleLock(DmGetResource(sysResTAppCode, 1));

			// There's extremely little stack space available when this
			// notification is sent, so a normal notification wouldn't work. 
			// Hence we use a callback notification.

			SysNotifyRegister(cardNo, dbID, sysNotifyResetFinishedEvent,
				ResetFinishedCallback, resetFinishedPriority, NULL);

			// NOTE: Normally an application that uses callback notifications
			// would have to protect itself and register for the
			// sysNotifyDeleteProtectedEvent notification. Because we're going to
			// deregister in a couple seconds, we don't need to worry about the
			// user deleting the app while it's still locked.
#endif
		}

		// Reset pen calibration information.

		if (OnPalmSim())
		{
			ResetCalibrationInfo();
		}

		SavePrefs(&prefs);
	}

	return errNone;
}


/******************************************************************************
 *
 * FUNCTION:	HandleLaunchOrCallApp
 *
 * DESCRIPTION:	Handle the launch code telling us to invoke an application
 *				using SysAppLaunch.
 *
 * PARAMETERS:	cmd - word value specifying the launch code. 
 *				cmdPB - pointer to a structure that is associated with the
 *					launch code. 
 *				launchFlags - word value providing extra information about the
 *					launch.
 *
 * RETURNED:	Result of launch
 *
 *****************************************************************************/

static UInt32 HandleSysAppLaunch(
	UInt16 cmd_ UNUSED, MemPtr cmdPBP, UInt16 launchFlags_ UNUSED)
{
	/*
		Read the data from the parameter block passed to us:
		
		Offset	Size	Description
		0		2		Value to pass as cardNo parameter
		2		2		Value to pass as launchFlags parameter
		4		2		Value to pass as cmd parameter
		6		n		Database name (NUL terminated)
		6+n		m		Any data to pass as cmdPBP parameter
	*/

#undef OFFSET_OF_CARDNO
#undef OFFSET_OF_LAUNCHFLAGS
#undef OFFSET_OF_CMD
#undef OFFSET_OF_DBNAME

#define OFFSET_OF_CARDNO		0
#define OFFSET_OF_LAUNCHFLAGS	(OFFSET_OF_CARDNO + 2)
#define OFFSET_OF_CMD			(OFFSET_OF_LAUNCHFLAGS + 2)
#define OFFSET_OF_DBNAME		(OFFSET_OF_CMD + 2)

	UInt16	cardNo		= Read16(cmdPBP, OFFSET_OF_CARDNO);
	UInt16	launchFlags	= Read16(cmdPBP, OFFSET_OF_LAUNCHFLAGS);
	UInt16	cmd			= Read16(cmdPBP, OFFSET_OF_CMD);
	Char*	dbName		= (Char*) cmdPBP + OFFSET_OF_DBNAME;
	LocalID	dbID		= DmFindDatabase(cardNo, dbName);
	MemPtr	subCmdPBP	= ReallocateCmdPBP(cmdPBP, OFFSET_OF_DBNAME + StrLen(dbName) + 1);
	UInt32	result;
	Err		err;

	if (dbID == 0)
	{
		return -1;	// What to return here?
	}

	if (subCmdPBP == (MemPtr) -1)
	{
		return -1;	// What to return here?
	}

	err = SysAppLaunch(cardNo, dbID, launchFlags, cmd, subCmdPBP, &result);

	// Free up the memory block we allocated.  However, if we launched the
	// application in its own thread, we should probably hold off on doing
	// that, as the new application may still be starting up.  In that case,
	// we just have to assume that the called application will dispose of
	// the block we sent it.

	if ((launchFlags & sysAppLaunchFlagNewThread) == 0)
	{
		MemPtrFree(subCmdPBP);
	}

	if (err != errNone)
	{
		return -1;	// What to return here?
	}

	return result;
}


/******************************************************************************
 *
 * FUNCTION:	HandleSysUIAppSwitch
 *
 * DESCRIPTION:	Handle the launch code telling us to invoke an application
 *				using SysUIAppSwitch.
 *
 * PARAMETERS:	cmd - word value specifying the launch code. 
 *				cmdPB - pointer to a structure that is associated with the
 *					launch code. 
 *				launchFlags - word value providing extra information about the
 *					launch.
 *
 * RETURNED:	Result of launch
 *
 *****************************************************************************/

static UInt32 HandleSysUIAppSwitch(
	UInt16 cmd_ UNUSED, MemPtr cmdPBP, UInt16 launchFlags_ UNUSED)
{
	/*
		Read the data from the parameter block passed to us:
		
		Offset	Size	Description
		0		2		Value to pass as cardNo parameter
		2		2		Value to pass as cmd parameter
		4		n		Database name (NUL terminated)
		4+n		m		Any data to pass as cmdPBP parameter
	*/

#undef OFFSET_OF_CARDNO
#undef OFFSET_OF_CMD
#undef OFFSET_OF_DBNAME

#define OFFSET_OF_CARDNO		0
#define OFFSET_OF_CMD			(OFFSET_OF_CARDNO + 2)
#define OFFSET_OF_DBNAME		(OFFSET_OF_CMD + 2)

	UInt16	cardNo		= Read16(cmdPBP, OFFSET_OF_CARDNO);
	UInt16	cmd			= Read16(cmdPBP, OFFSET_OF_CMD);
	Char*	dbName		= (Char*) cmdPBP + OFFSET_OF_DBNAME;
	LocalID	dbID		= DmFindDatabase(cardNo, dbName);
	MemPtr	subCmdPBP	= ReallocateCmdPBP(cmdPBP, OFFSET_OF_DBNAME + StrLen(dbName) + 1);
	Err		err;
	
	if (dbID == 0)
	{
		return -1;	// What to return here?
	}

	if (subCmdPBP == (MemPtr) -1)
	{
		return -1;	// What to return here?
	}

	err = SysUIAppSwitch(cardNo, dbID, cmd, subCmdPBP);
	if (err != errNone)
	{
		return -1;	// What to return here?
	}

	return 0;	
}


/******************************************************************************
 *
 * FUNCTION:	HandleDisplayAlarm
 *
 * DESCRIPTION:	Handle the launch code telling us that the alarm we set was
 *				triggered.  Here, we relaunch ourself in a more UI-friendly
 *				context so that we can show a message to the user.
 *
 * PARAMETERS:	cmd - word value specifying the launch code. 
 *				cmdPB - pointer to a structure that is associated with the
 *					launch code. 
 *				launchFlags - word value providing extra information about the
 *					launch.
 *
 * RETURNED:	Result of launch
 *
 *****************************************************************************/

#if NOTIFY_DEBUG_CONSOLE
static UInt32 HandleDisplayAlarm(
	UInt16 cmd UNUSED, MemPtr cmdPBP, UInt16 launchFlags UNUSED)
{
	Err error = errNone;
	SysAlarmTriggeredParamType* pAlarmParams;

	pAlarmParams = (SysAlarmTriggeredParamType*) cmdPBP;

	switch (pAlarmParams->ref)
	{
		case onResetAlarmRef:
		{
			UInt16 cardNo;
			LocalID dbID;

			// Unlock the code resource that we locked when registering for the
			// callback notification.

			SysCurAppDatabase(&cardNo, &dbID);
			SysNotifyUnregister(cardNo, dbID, sysNotifyResetFinishedEvent, resetFinishedPriority);
			MemHandleUnlock(DmGetResource(sysResTAppCode, 1));

			// We could safely display an alert at this point to warn the user,
			// but the console task in Palm OS stalls during the
			// sysAppLaunchCmdDisplayAlarm launch code. That prevents
			// debugger connections, so it would be bad to linger in this
			// launch code. Instead, we switch the current app to DevNub with a
			// special launch code telling it to display the warning and then
			// exit.

			error = SysUIAppSwitch(cardNo, dbID, displayWarningLaunchCmd, NULL);

			break;
		}
	}

	return error;
}
#endif


/******************************************************************************
 *
 * FUNCTION:	HandleDisplayWarningLaunchCmd
 *
 * DESCRIPTION:	Handle our custom launch code telling us to display a warning
 *				dialog that the debug console was launched.
 *
 * PARAMETERS:	cmd - word value specifying the launch code. 
 *				cmdPB - pointer to a structure that is associated with the
 *					launch code. 
 *				launchFlags - word value providing extra information about the
 *					launch.
 *
 * RETURNED:	Result of launch
 *
 *****************************************************************************/

#if NOTIFY_DEBUG_CONSOLE
static UInt32 HandleDisplayWarningLaunchCmd(
	UInt16 cmd UNUSED, MemPtr cmdPBP UNUSED, UInt16 launchFlags UNUSED)
{
	// To actually warn the user, we show an alert.
	
	switch (FrmAlert(OnResetAlert))
	{
		case OnResetOK:
			// Nothing to do except exit.
			break;

		case OnResetCancel:
		{
			// Turn off the on reset option.

			DevNubPreferenceType prefs;
			GetPrefs(&prefs);
			prefs.enterDebugConsoleOnReset = false;
			SavePrefs(&prefs);

			// Reset to exit console mode.

			SysReset();
			break;
		}
	}
	
	return errNone;
}
#endif


/******************************************************************************
 *
 * FUNCTION:	PilotMain
 *
 * DESCRIPTION:	This is the main entry point for the application.
 *
 * PARAMETERS:	cmd - word value specifying the launch code. 
 *				cmdPB - pointer to a structure that is associated with the
 *					launch code. 
 *				launchFlags - word value providing extra information about the
 *					launch.
 *
 * RETURNED:	Result of launch
 *
 *****************************************************************************/

UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
	UInt32 result = 0;

	switch (cmd)
	{
		case sysAppLaunchCmdNormalLaunch:
  			result = HandleNormalLaunch(cmd, cmdPBP, launchFlags);
			break;

		case sysAppLaunchCmdSystemReset:
			result = HandleSystemReset(cmd, cmdPBP, launchFlags);
			break;

		case sysAppLaunch:
			result = HandleSysAppLaunch(cmd, cmdPBP, launchFlags);
			break;

		case sysUIAppSwitch:
			result = HandleSysUIAppSwitch(cmd, cmdPBP, launchFlags);
			break;

#if NOTIFY_DEBUG_CONSOLE
		case sysAppLaunchCmdAlarmTriggered:
			// We want to warn the user, but UI isn't allowed during this
			// notification. Wait for the sysAppLaunchCmdDisplayAlarm that
			// follows.
			break;

		case sysAppLaunchCmdDisplayAlarm:
			result = HandleDisplayAlarm(cmd, cmdPBP, launchFlags);
			break;

		case displayWarningLaunchCmd:
			result = HandleDisplayWarningLaunchCmd(cmd, cmdPBP, launchFlags);
			break;
#endif
	}

	return result;
}
